<?php
include_once '../carabi/function/utilities.inc';
include_once '../carabi/classes/class.utls.php';
/**
* Документ из БД, представленный различными полями различных записей, 
* для редактирования в Web-формах
*
* @author aleksandr
*/
class Document {
	public $id; //ID главной записи, от которой строится документ
	public $formName; //название формы в БД, по которой строится документ (структура объектов и полей)
	public $fields; //ассоциативный массив полей, входящих в документ
	public $allFieldsEntered; //все требуемые поля заполнены пользователем
	public $errorMessage = "Вы не заполнили все обязательные поля!"; //сообщение, выводимое при ошибке, если не все поля заполнены
	public $wasRecentlySaved; //
	public $savedMessage = "Документ сохранён"; //сообщение, выводимое после сохранения документа

	public function __construct($id, $formName, $fields = null) {
		$this->id = $id;
		$this->formName = $formName;
		if ($fields == null) {
			$this->fields = array();
		} else {
			$this->fields = $fields;
		}
		$this->allFieldsEntered = true;
	}

	/**
	* Возвращение поля по имени
	* @param string Имя поля
	* @return Field 
	*/
	public function __get($name) {
		if (isset($this->fields[$name])) {
			return $this->fields[$name];
		} else {
			throw new Exception("Документ {$this->formName} с ID {$this->id} не содержит поля $name");
		}
	}

	/**
	* Сохранение поля по имени
	* @param String $name
	* @param String $value 
	*/
	public function __set ($name, $value) {
		if (!isset($this->fields[$name])) {
			$this->fields[$name] = new Field();
		}
		$field = $this->fields[$name];
		$field->fieldName = $this->formName . "_" . $name;
		$field->value = $value;
		$field->toWrite = true;
		$this->allFieldsEntered &= $this->checkFieldEntered($field);
	}

	/**
	* Добавление поля в документ
	* @param Field $field
	*/
	public function addField(Field $field) {
		$this->fields[$field->fieldName] = $field;
	}

	/**
	* Сохранение значения поля, полученного из Web-формы
	* @param string $name Имя поля (одно в форме и в БД)
	*/
	public function saveWebPostField($fieldName, $iconv = false) {
		$value = get($this->formName . "_" . $fieldName);
		if ($iconv) {
			$value = iconv("UTF-8", "CP1251", $value);
		}
		$this->__set($fieldName, $value);
	}

	/**
	* Сохранение ряда полей, полученных из Web-формы или всего документа
	* @param array $names массив имён полей
	*/
	public function saveWebPostFields($fieldNames = null) {
		if (isset ($fieldNames)) {
			foreach ($fieldNames as $fieldName) {
				$this->__set($fieldName, get($this->formName . "_" . $fieldName));
			}
		} else {
			foreach ($this->fields as $field) {
				$field->value = get($field->fieldName);
				$field->toWrite = true;
				$this->allFieldsEntered &= $this->checkFieldEntered($field);
			}
		}
	}

	/**
	* Поле заполнено верно (либо его не требуется заполнять)
	* @param string $fieldName
	* @return boolean 
	*/
	private function checkFieldEntered($field) {
		$entered = $field->value;
		$wasEntered = $entered !== null && $entered != "";
		//поле считается введённым, если может быть пустым или было введено
		$enteredCorrectly = $wasEntered || !$field->notNull;
		$field->reenter = !$enteredCorrectly;
		return $enteredCorrectly;
	}

	public static function __set_state($array) {
		$document = new Document(null, null, null);
		$document->id = $array["id"];
		$document->formName = $array["formName"];
		$document->fields = $array["fields"];
		return $document;
	}

	public function save() {
		return WaveEditor::saveDocument($this);
	}

	public function setFormName($formName) {
		$this->formName = $formName;
		foreach ($this->fields as $fieldName => $field) {
			$field->fieldName = $formName . "_" . $fieldName;
		}
	}

	/**
	* Инициализация web-шаблона уже введёнными значениями перед повторной
	* попыткой ввода при ошибке пользователя
	*/
	public function tryAgain() {
		foreach ($this->fields as $field) {
			if (isset($field->readOnly) && $field->readOnly == 0) {
				$field->value = get($field->fieldName);
			}
		}
	}

	public function updateAllFieldsEntered() {
		$this->allFieldsEntered = true;
		foreach ($this->fields as $field) {
			$this->allFieldsEntered &= !$field->reenter;
		}
		return $this->allFieldsEntered;
	}

	/**Если документ недавно сохранялся, об этом упоминается в сессии для вывода
	* сообщения пользователю. При первом чтении запись из сессии удаляется.
	* 
	*/
	public function checkRecentSaving() {
		if (is_array($_SESSION["savedDocuments"]) && in_array($this->id, $_SESSION["savedDocuments"])) {
			$this->wasRecentlySaved = true;
			array_splice($_SESSION["savedDocuments"], array_search($document->id, $_SESSION["savedDocuments"]), 1);
		}
	}

	/**
	* выдаёт содержимое документа в виде строки имя_поля=значение&имя=значение&
	* @return string
	*/
	public function serialize() {
		$serial = "";
		foreach ($this->fields as $field) {
			$serial .= "{$field->fieldName}={$field->value}&";
		}
		return $serial;
	}
}

?>
